#!/usr/bin/env python3

"""
Copyright (C) 2020  Aidan Webster

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
"""

import time
import os
import base64
import re

from PyQt5.QtCore import *
from PyQt5.QtGui import *
from PyQt5.QtWidgets import *
from passlib.hash import argon2
from cryptography.fernet import Fernet
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.kdf.pbkdf2 import PBKDF2HMAC


#App class to hold the initial QApplication and system info
class App:
    def __init__(self):
        self.app = QApplication([])
        self.screenRes = self.app.desktop().availableGeometry()
        #!!DETERMINE ROUNDS AT INSTALL AND SAVE IN CONFIG FILE!!
        self.ROUNDS = 14

    def getScreenWidth(self):
        return self.screenRes.width()

    def getScreenHeight(self):
        return self.screenRes.height()

    def getRounds(self):
        return self.ROUNDS

    def run(self):
        self.app.exec()

    def forceUpdate(self):
        self.app.processEvents()


#Window class to create a window
#Note: This class is not extensive, it only includes methods for widgets used
class Window:
    def __init__(self, title, width=None, height=None, icon=""):
        self.width          = width
        self.height         = height
        self.labels         = {}
        self.pushButtons    = {}
        self.dropdowns      = {}
        self.lineEdits      = {}
        self.plainTextEdits = {}
        self.tables         = {}
        self.window         = QWidget()

        self.window.setWindowTitle(title)
        self.window.setWindowIcon(QIcon(icon))

        if width or height:
            self.window.resize(width, height)

        self.layout = QGridLayout()

    #Add a label widget to the window
    def addLabel(self, id, row, column, rowSpan=1, columnSpan=1, pixmap=False,
                    image="", align="", text="", font="Arial", size=10):
        self.labels[id] = QLabel()

        if pixmap:
            pixmap = QPixmap(image)
            pixmap = pixmap.scaledToWidth(self.width - 25)
            self.labels[id].setPixmap(pixmap)
        else:
            self.labels[id].setFont(QFont(font, size))
            self.labels[id].setText(text)

        self.labels[id].setAlignment(eval(align))

        self.layout.addWidget(self.labels[id], row, column, rowSpan, columnSpan)

    #Add a push button widget to the window
    def addPushButton(self, id, text, row, column, rowSpan=1, columnSpan=1,
                        action="", font="Arial", size=10):
        self.pushButtons[id] = QPushButton(text)

        self.pushButtons[id].setFont(QFont(font, size))

        if len(action) > 0:
            self.pushButtons[id].clicked.connect(eval(action))

        self.layout.addWidget(self.pushButtons[id], row, column,
                                rowSpan, columnSpan)

    #Add a dropdown widget to the window
    def addDropdown(self, id, row, column, items, rowSpan=1, columnSpan=1,
                        font="Arial", size=10):
        self.dropdowns[id] = QComboBox()

        self.dropdowns[id].setFont(QFont(font, size))
        self.dropdowns[id].addItems(items)

        self.layout.addWidget(self.dropdowns[id], row, column,
                                rowSpan, columnSpan)

    #Add a line edit widget to the window
    def addLineEdit(self, id, row, column, rowSpan=1, columnSpan=1,
                        placeholder="", password=False, font="Arial", size=10):
        self.lineEdits[id] = QLineEdit()

        self.lineEdits[id].setFont(QFont(font, size))

        self.lineEdits[id].setPlaceholderText(placeholder)

        if password:
            self.lineEdits[id].setEchoMode(QLineEdit.Password)

        self.layout.addWidget(self.lineEdits[id], row, column,
                                rowSpan, columnSpan)

    #Add a plain text edit to the window
    def addPlainTextEdit(self, id, row, column, rowSpan=1, columnSpan=1,
                            placeholder="", height=90, font="Arial", size=10):
        self.plainTextEdits[id] = QPlainTextEdit()

        self.plainTextEdits[id].setFont(QFont(font, size))
        self.plainTextEdits[id].setPlaceholderText(placeholder)
        self.plainTextEdits[id].setFixedHeight(height)

        self.layout.addWidget(self.plainTextEdits[id], row, column,
                                rowSpan, columnSpan)

    #Add a table widget to the window
    def addTable(self, id, row, column, rowSpan=1, columnSpan=1,
                    height=0, width=0, readOnly=False, horizontalHeaders=None,
                    verticalHeaders=None, horizontalResize="ResizeToContents",
                    verticalResize="ResizeToContents", stretchLastCol=False,
                    clickConnect=None, wordWrap=True, font="Arial", size=10):
        self.tables[id] = QTableWidget()

        self.tables[id].setFont(QFont(font, size))
        self.tables[id].setRowCount(height)
        self.tables[id].setColumnCount(width)

        #Check that resize modes are valid
        validModes = [
                        "Interactive",
                        "Fixed",
                        "Stretch",
                        "ResizeToContents",
                        "Custom"
                        ]
        for i in range(0, len(validModes)):
            if horizontalResize == validModes[i]:
                validHorizontal = True

            if verticalResize == validModes[i]:
                validVertical = True

        if not validHorizontal or not validVertical:
            print("Invalid horizontalResize or verticalResize")
            print("Valid options are: Interactive, Fixed, Stretch, ResizeToContents or Custom")
            print("Fatal Error, exiting...")
            exit()

        #Set resize modes horizontally and vertically
        horizontalHeader = self.tables[id].horizontalHeader()
        verticalHeader   = self.tables[id].verticalHeader()
        horizontalResize   = "QHeaderView." + horizontalResize
        verticalResize   = "QHeaderView." + verticalResize
        horizontalHeader.setSectionResizeMode(eval(horizontalResize))
        verticalHeader.setSectionResizeMode(eval(verticalResize))

        #Disable horizontal headers if no values otherwise set values
        if horizontalHeaders != None:
            self.tables[id].setHorizontalHeaderLabels(horizontalHeaders)
        else:
            self.tables[id].horizontalHeader().hide()

        #Disable vertical headers if no values otherwise set values
        if verticalHeaders != None:
            self.tables[id].setVerticalHeaderLabels(verticalHeaders)
        else:
            self.tables[id].verticalHeader().hide()

        #Enable word wrap if true
        if wordWrap:
            self.tables[id].setWordWrap(True)

        #Enable read-only if true
        if readOnly:
            self.tables[id].setEditTriggers(QAbstractItemView.NoEditTriggers)

        if stretchLastCol:
            self.tables[id].horizontalHeader().setStretchLastSection(True)

        #Connect cell click action to function
        if clickConnect != None:
            self.tables[id].cellClicked.connect(eval(clickConnect))

        self.layout.addWidget(self.tables[id], row, column, rowSpan, columnSpan)

    #Set layout and show window
    def show(self):
        self.window.setLayout(self.layout)
        self.window.show()

    #Clear the window (i.e. remove all widgets) and wipe them from RAM
    def clear(self):
        for i in reversed(range(self.layout.count())):
            widget = self.layout.itemAt(i).widget()
            self.layout.removeWidget(widget)
            widget.setParent(None)

        for i in range(0, len(self.labels)):
            self.labels[i] = None

        for i in range(0, len(self.pushButtons)):
            self.pushButtons[i] = None

        for i in range(0, len(self.dropdowns)):
            self.dropdowns[i] = None

        for i in range(0, len(self.lineEdits)):
            self.lineEdits[i] = None

        for i in range(0, len(self.plainTextEdits)):
            self.plainTextEdits[i] = None

        for i in range(0, len(self.tables)):
            self.tables[i] = None


#DataGuardWindow class extends Window class and adds methods to create windows
#used in the program
class DataGuardWindow(Window):
    #Reconfigure window for registration
    def register(self, error=None, username=None, passwd=None, passwdConf=None):
        #Clear window, resize and centre on screen
        self.clear()
        self.window.setWindowState(Qt.WindowNoState)
        #It takes two for this to work, IDK why but if not the width is too big
        self.window.resize(450, 454)
        self.window.resize(450, 454)
        self.window.move(int((app.getScreenWidth() - 450)/2), int((app.getScreenHeight() - 450)/2))

        #Add widgets
        self.addLabel("logo", 0, 0, columnSpan=2, pixmap=True,
                            image="bitmaps/logo.png",
                            align="Qt.AlignHCenter | Qt.AlignTop")
        self.addLineEdit("regUsernameInput", 1, 0, placeholder="Username",
                                columnSpan=2, size=18)
        self.addLineEdit("regPasswordInput", 2, 0, placeholder="Password",
                                password=True, columnSpan=2, size=18)
        self.addLineEdit("regPasswordConfirmInput", 3, 0,
                                placeholder="Confirm Password",
                                password=True, columnSpan=2, size=18)
        self.addPushButton("regBackButton", "Back", 4, 0,
                                action="window.login", size=18)
        self.addPushButton("regRegisterButton", "Register", 4, 1,
                                action="user.register", size=18)

        #Run register method if enter pressed in line edit
        self.lineEdits["regUsernameInput"].returnPressed.connect(user.register)
        self.lineEdits["regPasswordInput"].returnPressed.connect(user.register)
        self.lineEdits["regPasswordConfirmInput"].returnPressed.connect(user.register)

        #Fill in fields with previous values
        self.lineEdits["regUsernameInput"].setText(username)
        self.lineEdits["regPasswordInput"].setText(passwd)
        self.lineEdits["regPasswordConfirmInput"].setText(passwdConf)

        #Set focus on the username input
        self.lineEdits["regUsernameInput"].setFocus()

        #Error checking
        if error == "noUsr":
            self.lineEdits["regUsernameInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["regUsernameInput"].setPlaceholderText("No Username Entered")
        elif error == "exists":
            self.lineEdits["regUsernameInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["regUsernameInput"].setPlaceholderText("Account Already Exists")
        elif error == "noPwd":
            self.lineEdits["regPasswordInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["regPasswordInput"].setPlaceholderText("No Password Entered")
        elif error == "noPwdMch":
            self.lineEdits["regPasswordConfirmInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["regPasswordConfirmInput"].setPlaceholderText("Passwords Didn't Match")

    #Reconfigure window for login
    def login(self, error=None, username=None):
        #Scan users directory to find all usernames
        users = [f.name for f in os.scandir("data/users/") if f.is_dir()]

        #If there are no users get into create account only mode
        if users == []:
            users = ["No Existing Accounts"]
            passwdPlaceholder = ""
            disable = True
        else:
            passwdPlaceholder = "Password"
            disable = False

        #Prepare the window for login mode
        self.clear()
        self.window.setWindowState(Qt.WindowNoState)
        #It takes two for this to work, IDK why but if not the width is too big
        self.window.resize(450, 450)
        self.window.resize(450, 450)
        self.window.move(int((app.getScreenWidth() - 450)/2), int((app.getScreenHeight() - 450)/2))

        #Add widgets
        self.addLabel("logo", 0, 0, columnSpan=2, pixmap=True,
                            image="bitmaps/logo.png",
                            align="Qt.AlignHCenter | Qt.AlignTop")

        self.addLineEdit("logPasswordInput", 2, 0, placeholder=passwdPlaceholder,
                                password=True, columnSpan=2, size=18)

        self.addDropdown("logUsernameDropdown", 1, 0, users,
                            columnSpan=2, size=18)

        self.addPushButton("logLoginButton", "Login", 3, 0,
                                action="user.login", size=18)

        self.addPushButton("logCreateAccountButton", "Create an Account", 3, 1,
                                action="window.register", size=18)

        self.addPushButton("logExitButton", "Exit", 4, 0, columnSpan=2,
                                action="exit", size=18)

        #Set value for the username input
        index = self.dropdowns["logUsernameDropdown"].findText(username)
        if index >= 0:
             self.dropdowns["logUsernameDropdown"].setCurrentIndex(index)

        #Set focus on the password input
        self.lineEdits["logPasswordInput"].setFocus()

        #If enter is pressed in the login line edit execute login method
        self.lineEdits["logPasswordInput"].returnPressed.connect(user.login)

        #Make buttons unclickable if disable got set (i.e. if no users found)
        if disable:
            self.enableMode("loginnousers", False)

        #Error checking
        if error == "noPwd":
            self.lineEdits["logPasswordInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["logPasswordInput"].setPlaceholderText("No Password Entered")
        elif error == "incorrectPwd":
            self.lineEdits["logPasswordInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["logPasswordInput"].setPlaceholderText("Password Incorrect")

    #Main window
    def main(self):
        #Clear the window, resize and centre on screen
        self.clear()
        self.window.setWindowState(Qt.WindowMaximized)

        #Define table headers
        tableHeaders = ["Account", "Username", "Email", "Password", "Notes"]

        #Add widgets
        window.addTable("mainPasswordsTable", 0, 0, width=5, columnSpan=6,
                            readOnly=True, horizontalHeaders=tableHeaders,
                            horizontalResize="ResizeToContents",
                            stretchLastCol=True,
                            clickConnect="self.cellClicked", size=14)
        window.addPushButton("mainAddButton", "Add Entry", 1, 0,
                                action="self.addEntry", size=18)
        window.addPushButton("mainEditEntButton", "Edit Entry", 1, 1,
                                action="self.editEntry", size=18)
        window.addPushButton("mainDelEntButton", "Delete Entry", 1, 2,
                                action="self.delEntry", size=18)
        window.addPushButton("mainDelAcctButton", "Delete Account", 1, 3,
                                action="self.delAccount", size=18)
        window.addPushButton("mainLogoutButton", "Logout", 1, 4,
                                action="user.logout", size=18)
        window.addPushButton("mainExitButton", "Exit", 1, 5,
                                action="exit", size=18)

        #Populate the table with accounts info, hiding passwords for now
        usernameDir = "data/users/" + user.getUsername() + "/accounts/"
        accounts = [f for f in os.listdir(usernameDir) if os.path.isfile(os.path.join(usernameDir, f))]
        for i in range(0, len(accounts)):
            rowPosition = self.tables["mainPasswordsTable"].rowCount()
            self.tables["mainPasswordsTable"].insertRow(rowPosition)
            accountFile = open("data/users/" + user.getUsername() + "/accounts/" + accounts[i], "r")
            accountLine = accountFile.readline().encode()
            accountFile.close()
            f = Fernet(user.getKey())
            accountLine = f.decrypt(accountLine).decode()
            accountLine = accountLine.splitlines()
            self.tables["mainPasswordsTable"].setItem(i, 0, QTableWidgetItem(accounts[i]))
            self.tables["mainPasswordsTable"].setItem(i, 1, QTableWidgetItem(accountLine[0]))
            self.tables["mainPasswordsTable"].setItem(i, 2, QTableWidgetItem(accountLine[1]))
            self.tables["mainPasswordsTable"].setItem(i, 3, QTableWidgetItem("••••••••••••"))
            #Grab all the lines for the notes section
            note = ""
            for j in range (3, len(accountLine)):
                note += accountLine[j] + "\n"
            self.tables["mainPasswordsTable"].setItem(i, 4, QTableWidgetItem(note))
            note = None

        #Resize rows to contents
        self.tables["mainPasswordsTable"].resizeRowsToContents()

        #Wipe variables for (some) security
        f = None
        accountLine = None

    #Prepare the window for add entry mode
    def addEntry(self, error=None, name=None, username=None, email=None,
                    password=None, notes=None):
        self.clear()
        self.window.setWindowState(Qt.WindowNoState)
        #It takes two for this to work, IDK why but if not the width is too big
        self.window.resize(450, 600)
        self.window.resize(450, 600)
        self.window.move(int((app.getScreenWidth() - 450)/2), int((app.getScreenHeight() - 600)/2))

        #Add widgets
        self.addLabel("logo", 0, 0, columnSpan=2, pixmap=True,
                            image="bitmaps/logo.png",
                            align="Qt.AlignHCenter | Qt.AlignTop")
        self.addLineEdit("entNameInput", 1, 0, placeholder="Account Name",
                                columnSpan=2, size=18)
        self.addLineEdit("entUsernameInput", 2, 0,
                                placeholder="Username (optional)", columnSpan=2,
                                size=18)
        self.addLineEdit("entEmailInput", 3, 0,
                                placeholder="Email (optional)", columnSpan=2,
                                size=18)
        self.addLineEdit("entPasswordInput", 4, 0, password=True,
                                placeholder="Password", columnSpan=2, size=18)
        self.addPlainTextEdit("entNotesInput", 5, 0, columnSpan=2,
                                    placeholder="Additional notes (optional)...",
                                    size=18)
        self.addPushButton("entBackButton", "Back", 6, 0,
                                action="window.main", size=18)
        self.addPushButton("entAddButton", "Add Entry", 6, 1,
                                action="user.addEntry", size=18)

        #Set values of previous entries
        self.lineEdits["entNameInput"].setText(name)
        self.lineEdits["entUsernameInput"].setText(username)
        self.lineEdits["entEmailInput"].setText(email)
        self.lineEdits["entPasswordInput"].setText(password)
        self.plainTextEdits["entNotesInput"].setPlainText(notes)

        #Add connections
        self.lineEdits["entNameInput"].returnPressed.connect(user.addEntry)
        self.lineEdits["entUsernameInput"].returnPressed.connect(user.addEntry)
        self.lineEdits["entEmailInput"].returnPressed.connect(user.addEntry)
        self.lineEdits["entPasswordInput"].returnPressed.connect(user.addEntry)

        #Error checking
        if error == "noName":
            self.lineEdits["entNameInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["entNameInput"].setPlaceholderText("No Name Entered")
        elif error == "entryExists":
            self.lineEdits["entNameInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["entNameInput"].setPlaceholderText("Entry Already Exists")
        elif error == "invalidEmail":
            self.lineEdits["entEmailInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["entEmailInput"].setPlaceholderText("Invalid Email Address")
        elif error == "noPwd":
            self.lineEdits["entPasswordInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["entPasswordInput"].setPlaceholderText("No Password Entered")

        #Set focus on the account name input
        self.lineEdits["entNameInput"].setFocus()

    #Prepare the window for edit entry mode
    def editEntry(self, error=None, selected=None, username=None, email=None,
                    password=None, notes=None):
        self.clear()
        self.window.setWindowState(Qt.WindowNoState)
        #It takes two for this to work, IDK why but if not the width is too big
        self.window.resize(450, 600)
        self.window.resize(450, 600)
        self.window.move(int((app.getScreenWidth() - 450)/2), int((app.getScreenHeight() - 600)/2))

        #Get list of accounts
        usernameDir = "data/users/" + user.getUsername() + "/accounts/"
        accounts = [f for f in os.listdir(usernameDir) if os.path.isfile(os.path.join(usernameDir, f))]
        accounts.insert(0, "Select an account...")

        #Add widgets
        self.addLabel("logo", 0, 0, columnSpan=2, pixmap=True,
                            image="bitmaps/logo.png",
                            align="Qt.AlignHCenter | Qt.AlignTop")
        self.addDropdown("editNameDropdown", 1, 0, accounts,
                            columnSpan=2, size=18)
        self.addLineEdit("editUsernameInput", 2, 0,
                            placeholder="Username (optional)",columnSpan=2,
                            size=18)
        self.addLineEdit("editEmailInput", 3, 0,
                                placeholder="Email (optional)", columnSpan=2, size=18)
        self.addLineEdit("editPasswordInput", 4, 0, password=True,
                                placeholder="Password", columnSpan=2, size=18)
        self.addPlainTextEdit("editNotesInput", 5, 0, columnSpan=2,
                                    placeholder="Additional notes (optional)...",
                                    size=18)
        self.addPushButton("editBackButton", "Back", 6, 0,
                                action="window.main", size=18)
        self.addPushButton("editUpdateButton", "Update Entry", 6, 1,
                                action="user.editEntry", size=18)

        #If account name dropdown changes call function to update others
        self.dropdowns["editNameDropdown"].currentTextChanged.connect(self.editUpdate)

        #Add connections
        self.lineEdits["editUsernameInput"].returnPressed.connect(user.editEntry)
        self.lineEdits["editEmailInput"].returnPressed.connect(user.editEntry)
        self.lineEdits["editPasswordInput"].returnPressed.connect(user.editEntry)


        #Disable the entries because we know "Select an account..." is selected
        self.enableMode("edit", False)

        #Set the value of the dropdown if specified
        if selected != None:
            index = self.dropdowns["editNameDropdown"].findText(selected)
            if index >= 0:
                 self.dropdowns["editNameDropdown"].setCurrentIndex(index)
            else:
                print("That entry doesn't exist. Exiting...")
                exit()

        #Set values of other inputs
        self.lineEdits["editUsernameInput"].setText(username)
        self.lineEdits["editEmailInput"].setText(email)
        self.lineEdits["editPasswordInput"].setText(password)
        if notes != None:
            self.lineEdits["editNotesInput"].setText(notes)

        #Check for previous errors
        if error == "noPassword":
            self.lineEdits["editPasswordInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["editPasswordInput"].setText("")
            self.lineEdits["editPasswordInput"].setPlaceholderText("No Password Entered")
        elif error == "invalidEmail":
            self.lineEdits["editEmailInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["editEmailInput"].setText("")
            self.lineEdits["editEmailInput"].setPlaceholderText("Invalid Email")

    #Delete an entry in the accounts table
    def delEntry(self):
        #Clear and resize window
        self.clear()
        self.window.setWindowState(Qt.WindowNoState)
        #It takes two for this to work, IDK why but if not the width is too big
        self.window.resize(450, 100)
        self.window.resize(450, 100)
        self.window.move(int((app.getScreenWidth() - 450)/2), int((app.getScreenHeight() - 100)/2))

        #Get list of accounts
        usernameDir = "data/users/" + user.getUsername() + "/accounts/"
        accounts = [f for f in os.listdir(usernameDir) if os.path.isfile(os.path.join(usernameDir, f))]
        accounts.insert(0, "Select an entry...")

        #Add widgets
        self.addDropdown("delNameDropdown", 1, 0, accounts,
                            columnSpan=2, size=18)
        self.addPushButton("delBackButton", "Back", 2, 0,
                                action="window.main", size=18)
        self.addPushButton("delEntryButton", "Delete Entry", 2, 1,
                                action="user.delEntry", size=18)

        #Check whether to enable or disable button when dropdown changed
        self.dropdowns["delNameDropdown"].currentTextChanged.connect(self.delEntryUpdate)

        #Disable button as we know "Select an entry..." is selected
        self.enableMode("delEntry", False)

    #Delete an account
    def delAccount(self, error=None):
        #Clear and resize window
        self.clear()
        self.window.setWindowState(Qt.WindowNoState)
        #It takes two for this to work, IDK why but if not the width is too big
        self.window.resize(450, 100)
        self.window.resize(450, 100)
        self.window.move(int((app.getScreenWidth() - 450)/2), int((app.getScreenHeight() - 100)/2))

        #Get list of accounts
        accounts = [f.name for f in os.scandir("data/users/") if f.is_dir()]
        accounts.insert(0, "Select an account...")

        #Add widgets
        self.addLineEdit("delAccPasswordInput", 0, 0, password=True,
                                placeholder="Type password to confirm...",
                                columnSpan=2, size=18)
        self.addPushButton("delAccBackButton", "Back", 1, 0,
                                action="window.main", size=18)
        self.addPushButton("delAccConfirmButton", "Delete Account", 1, 1,
                                action="user.delAccount", size=18)

        #Run delete method if enter pressed
        self.lineEdits["delAccPasswordInput"].returnPressed.connect(user.delAccount)

        #Error checking
        if error == "incorrectPwd":
            self.lineEdits["delAccPasswordInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["delAccPasswordInput"].setPlaceholderText("Password Incorrect")
        elif error == "noPwd":
            self.lineEdits["delAccPasswordInput"].setStyleSheet(
                        "border: 2px solid red; background: white; color: red;")
            self.lineEdits["delAccPasswordInput"].setPlaceholderText("No Password Entered")

        #Set focus on the password input
        self.lineEdits["delAccPasswordInput"].setFocus()

    #Updates widget values to match account when in edit account mode
    def editUpdate(self):
        #Read value of account dropdown
        accountName = self.dropdowns["editNameDropdown"].currentText()

        if accountName != "Select an account...":
            #Load in info from the account file
            accountFile = open("data/users/" + user.getUsername() + "/accounts/" + accountName, "r")
            accountLine = accountFile.readline().encode()
            accountFile.close()
            f = Fernet(user.getKey())
            accountLine = f.decrypt(accountLine).decode()
            accountLine = accountLine.splitlines()

            #Set values of entries
            self.lineEdits["editUsernameInput"].setText(accountLine[0])
            self.lineEdits["editEmailInput"].setText(accountLine[1])
            self.lineEdits["editPasswordInput"].setText(accountLine[2])
            self.plainTextEdits["editNotesInput"].setPlainText(accountLine[3])

            #Enable the entries
            self.enableMode("edit", True)

            #Set focus on the username input
            self.lineEdits["editUsernameInput"].setFocus()
        else:
            self.lineEdits["editUsernameInput"].setText("")
            self.lineEdits["editEmailInput"].setText("")
            self.lineEdits["editPasswordInput"].setText("")
            self.plainTextEdits["editNotesInput"].setPlainText("")
            self.enableMode("edit", False)

    def delEntryUpdate(self):
        #Read value of entry dropdown
        entryName = self.dropdowns["delNameDropdown"].currentText()

        if entryName != "Select an entry...":
            self.enableMode("delEntry", True)
        else:
            self.enableMode("delEntry", False)

    #Shows plaintext password when password cell is clicked
    def cellClicked(self, row, column):
        if column == 3:
            #Get text from accounts column on row of clicked cell
            accountName = self.tables["mainPasswordsTable"].item(row, 0).text()
            if accountName != None:
                #Get list of all files in 'accounts' directory
                usernameDir = "data/users/" + user.getUsername() + "/accounts/"
                accounts = [f for f in os.listdir(usernameDir) if os.path.isfile(os.path.join(usernameDir, f))]
                found = False
                for i in range(0, len(accounts)):
                    if accounts[i] == accountName:
                        found = True
                        number = i
                if found != True:
                    print("An unknown error occured. Exiting...")
                    exit()
                else:
                    #Decrypt password and display
                    accountFile = open("data/users/" + user.getUsername() + "/accounts/" + accounts[number], "r")
                    accountLine = accountFile.readline().encode()
                    accountFile.close()
                    f = Fernet(user.getKey())
                    accountLine = f.decrypt(accountLine).decode()
                    accountLine = accountLine.splitlines()
                    for i in range(0, len(accounts)):
                        self.tables["mainPasswordsTable"].setItem(i, 3, QTableWidgetItem("••••••••••••"))
                    self.tables["mainPasswordsTable"].setItem(number, 3, QTableWidgetItem(accountLine[2]))
        else:
            #Hide password again when unclicked
            usernameDir = "data/users/" + user.getUsername() + "/accounts/"
            accounts = [f for f in os.listdir(usernameDir) if os.path.isfile(os.path.join(usernameDir, f))]
            for i in range(0, len(accounts)):
                self.tables["mainPasswordsTable"].setItem(i, 3, QTableWidgetItem("••••••••••••"))

    #Disable certain widgets depending on windowMode
    def enableMode(self, windowMode, option):
        if windowMode.lower() == "register":
            self.pushButtons["regRegisterButton"].setText("Creating...")
            self.pushButtons["regBackButton"].setEnabled(option)
            self.lineEdits["regUsernameInput"].setEnabled(option)
            self.lineEdits["regPasswordInput"].setEnabled(option)
            self.lineEdits["regPasswordConfirmInput"].setEnabled(option)
        elif windowMode.lower() == "login":
            self.pushButtons["logLoginButton"].setText("Checking...")
            self.pushButtons["logCreateAccountButton"].setEnabled(option)
            self.pushButtons["logExitButton"].setEnabled(option)
            self.dropdowns["logUsernameDropdown"].setEnabled(option)
            self.lineEdits["logPasswordInput"].setEnabled(option)
        elif windowMode.lower() == "loginnousers":
            self.dropdowns["logUsernameDropdown"].setEnabled(option)
            self.lineEdits["logPasswordInput"].setEnabled(option)
            self.pushButtons["logLoginButton"].setEnabled(option)
        elif windowMode.lower() == "entry":
            self.pushButtons["entAddButton"].setText("Adding...")
            self.lineEdits["entNameInput"].setEnabled(option)
            self.lineEdits["entUsernameInput"].setEnabled(option)
            self.lineEdits["entEmailInput"].setEnabled(option)
            self.lineEdits["entPasswordInput"].setEnabled(option)
            self.plainTextEdits["entNotesInput"].setEnabled(option)
            self.pushButtons["entBackButton"].setEnabled(option)
        elif windowMode.lower() == "edit":
            self.lineEdits["editUsernameInput"].setEnabled(option)
            self.lineEdits["editEmailInput"].setEnabled(option)
            self.lineEdits["editPasswordInput"].setEnabled(option)
            self.plainTextEdits["editNotesInput"].setEnabled(option)
            self.pushButtons["editUpdateButton"].setEnabled(option)
        elif windowMode.lower() == "delentry":
            self.pushButtons["delEntryButton"].setEnabled(option)
        elif windowMode.lower() == "delaccount":
            self.pushButtons["delAccConfirmButton"].setText("Deleting...")
            self.lineEdits["delAccPasswordInput"].setEnabled(option)
            self.pushButtons["delAccBackButton"].setEnabled(option)
        else:
            print("Invalid windowMode for Window.enableMode! Exiting...")
            exit()

        #Force the window to update (not ideal but works)
        app.forceUpdate()

    def getData(self, windowMode):
        if windowMode.lower() == "register":
            #Return dictionary containing username, password and password conf
            username = self.lineEdits["regUsernameInput"].text()
            password = self.lineEdits["regPasswordInput"].text()
            passwordConf = self.lineEdits["regPasswordConfirmInput"].text()
            return {"username": username,
                    "password": password,
                    "passwordConf": passwordConf}
        elif windowMode.lower() == "login":
            #Return dictionary containing username and password
            username = self.dropdowns["logUsernameDropdown"].currentText()
            password = self.lineEdits["logPasswordInput"].text()
            return {"username": username, "password": password}
        elif windowMode.lower() == "entry":
            name = self.lineEdits["entNameInput"].text()
            username = self.lineEdits["entUsernameInput"].text()
            email = self.lineEdits["entEmailInput"].text()
            password = self.lineEdits["entPasswordInput"].text()
            notes = self.plainTextEdits["entNotesInput"].toPlainText()
            return {"name": name,
                    "username": username,
                    "email": email,
                    "password": password,
                    "notes": notes}
        elif windowMode.lower() == "edit":
            name = self.dropdowns["editNameDropdown"].currentText()
            username = self.lineEdits["editUsernameInput"].text()
            email = self.lineEdits["editEmailInput"].text()
            password = self.lineEdits["editPasswordInput"].text()
            notes = self.plainTextEdits["editNotesInput"].toPlainText()
            return {"name": name,
                    "username": username,
                    "email": email,
                    "password": password,
                    "notes": notes}
        elif windowMode.lower() == "delentry":
            name = self.dropdowns["delNameDropdown"].currentText()
            return name
        elif windowMode.lower() == "delaccount":
            password = self.lineEdits["delAccPasswordInput"].text()
            return password
        else:
            print("Invalid windowMode for Window.getData! Exiting...")
            exit()


#User class stores user info
class User:
    def __init__(self):
        self.username = None
        self.key = None

    #Create a new user account
    def register(self):
        #Disable all text entry during the registration process
        window.enableMode("register", False)

        #Grab data from the window
        username     = window.getData("register")["username"]
        password     = window.getData("register")["password"]
        passwordConf = window.getData("register")["passwordConf"]

        #Verify that everything's good with inputs
        if username == "":
            window.register(error="noUsr", passwd=password, passwdConf=passwordConf)
        elif os.path.isdir("data/users/" + username):
            window.register(error="exists", passwd=password, passwdConf=passwordConf)
        elif password == "":
            window.register(error="noPwd", username=username, passwdConf=passwordConf)
        elif password != passwordConf:
            window.register(error="noPwdMch", username=username, passwd=password)
        else:
            #If all is good start registering account; make user directory
            os.makedirs("data/users/" + username + "/accounts")

            #Create and open user password file in write mode
            userFile = open(r"data/users/" + username + "/.meta", "w")

            #Hash password with Argon2 and write to password file
            hashedPw  = argon2.using(rounds=app.getRounds()).hash(password)
            userFile.write(hashedPw)
            userFile.close()

            #Return to login window
            window.login()

    #Verify the password entered is correct and generate encryption key
    def login(self):
        #Disable all text entry during the login process
        window.enableMode("login", False)

        #Store username for later use
        self.username = window.getData("login")["username"]
        #Temporarily set password, this is wiped later for (some) security
        password = window.getData("login")["password"]

        #If password wasn't set, return to the login window with an error
        if password == "":
            window.login(error="noPwd", username=self.username)
        else:
            #Open the user's file read-only to check their password
            userFile = open("data/users/" + self.username + "/.meta", "r")
            #Grab that encrypted password
            pLine = userFile.readline().rstrip()
            userFile.close()

            #If the password matches let 'em in
            if argon2.verify(password, pLine):
                #Use a crappy salt because it needs to be the same each time
                #(Doing this a better way would make for a good update)
                salt = "2sWy6Fhj4" + self.username + "5mzFlcUAmVF"

                #Prepare for hashing to generate encryption key
                kdf = PBKDF2HMAC(
                    algorithm=hashes.SHA256(),
                    length=32,
                    salt=salt.encode(),
                    iterations=100000,
                    backend=default_backend()
                )

                #Generate and save the encryption key and wipe password var
                self.key = base64.urlsafe_b64encode(kdf.derive(password.encode()))
                password = None
                window.main()
            else:
                #If password verification failed return to login window with
                #password error
                password = None
                window.login(error="incorrectPwd", username=self.username)

    #Wipe username and key and return to login window
    def logout(self):
        self.username = None
        self.key      = None
        window.login()

    #Add an entry to the accounts table
    def addEntry(self):
        #Disable widgets during adding process
        window.enableMode("entry", False)

        #Regex to check if email is valid (RFC 5322 Official Standard)
        regex = r"(^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$)"

        #Get data from the line edits
        name = window.getData("entry")["name"]
        username = window.getData("entry")["username"]
        email = window.getData("entry")["email"]
        password = window.getData("entry")["password"]
        notes = window.getData("entry")["notes"]

        #Verify the data entered
        if username == "":
            username = "-"

        if email == "":
            email = "-"

        if notes == "":
            notes = "-"

        if name == "":
            window.addEntry(error="noName", username=username, email=email,
                                password=password, notes=notes)
        elif os.path.exists("data/users/" + self.username + "/accounts/" + name):
            window.addEntry(error="entryExists", username=username, email=email,
                                password=password, notes=notes)
        elif email != "-" and not re.fullmatch(regex, email):
            window.addEntry(error="invalidEmail", name=name, username=username,
                                password=password, notes=notes)
        elif password == "":
            window.addEntry(error="noPwd", name=name, username=username,
                                email=email, notes=notes)
        else:
            #Encrypt the info
            f = Fernet(self.getKey())
            text = username + "\n" + email + "\n" + password + "\n" + notes

            #Save the encrypted info to its file
            accountFile = open("data/users/" + self.username + "/accounts/" + name, "w")
            accountFile.write(f.encrypt(text.encode()).decode())
            accountFile.close()
            window.main()

    def editEntry(self):
        #Disable widgets during updating process
        window.enableMode("edit", False)

        #Regex to check if email is valid (RFC 5322 Official Standard)
        regex = r"(^[a-zA-Z0-9_.+-]+@[a-zA-Z0-9-]+\.[a-zA-Z0-9-.]+$)"

        #Get data from the line edits
        name = window.getData("edit")["name"]
        username = window.getData("edit")["username"]
        email = window.getData("edit")["email"]
        password = window.getData("edit")["password"]
        notes = window.getData("edit")["notes"]

        #Verify the data entered
        if username == "":
            username = "-"

        if email == "":
            email = "-"

        if notes == "":
            notes = "-"

        if email != "-" and not re.fullmatch(regex, email):
            window.editEntry(error="invalidEmail",
                                selected=name,
                                username=username,
                                password=password,
                                notes=notes
                                )
        elif password == "":
            window.editEntry(error="noPassword", selected=name,
                                username=username, email=email, notes=notes)
        else:
            #Encrypt the info
            f = Fernet(self.getKey())
            text = username + "\n" + email + "\n" + password + "\n" + notes

            #Save the encrypted info to its file
            accountFile = open("data/users/" + self.username + "/accounts/" + name, "w")
            #Truncate to 0 length (i.e. wipe the file so we don't make a mess)
            accountFile.truncate(0)
            accountFile.write(f.encrypt(text.encode()).decode())
            accountFile.close()
            window.main()

    def delEntry(self):
        name = window.getData("delentry")
        os.remove("data/users/" + self.username + "/accounts/" + name)
        window.main()

    def delAccount(self):
        #Configure the window for account deletion
        window.enableMode("delaccount", False)
        #Get the password entered
        password = window.getData("delaccount")

        #Open the user's file read-only to check their password
        userFile = open("data/users/" + self.username + "/.meta", "r")
        #Grab that encrypted password
        pLine = userFile.readline().rstrip()
        userFile.close()

        #If no password entered display noPwd message
        if password == "":
            window.delAccount(error="noPwd")
        #If the password matches let 'em in
        elif argon2.verify(password, pLine):
            #Empty the user's directory
            for root, dirs, files in os.walk("data/users/" + self.username, topdown=False):
                for name in files:
                    os.remove(os.path.join(root, name))
                for name in dirs:
                    os.rmdir(os.path.join(root, name))

            #Delete the user's directory
            os.rmdir("data/users/" + self.username)

            #Log the current user out
            self.logout()
        #If password doesn't match and is not empty display inccrectPwd message
        else:
            window.delAccount(error="incorrectPwd")

    #Getters to return various variables
    def getUsername(self):
        return self.username

    def getKey(self):
        return self.key


#Main Program
app = App()
user = User()
window = DataGuardWindow("Data Guard", width=450, height=450, icon="bitmaps/icon.png")
window.login()
window.show()
app.run()
